<?php
/**
 * Plugin Name:  Push Post to MaxGood.work
 * Description:  Automatically pushes newly published WordPress posts (HTML + metadata, no binaries) to a MaxGood.work Avatar’s “Expertise”. Requires an existing Avatar/Organization on MaxGood.work. Learn more at https://maxgood.work
 * Version:      1.0.1
 * Author:       MaxGood.work
 * License:      GPL-2.0-or-later
 * License URI:  https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:  push-post-to-maxgoodwork
 */

if (!defined('ABSPATH')) exit;

const MGW_PUSH_OPTION = 'mgw_push_options';
const MGW_API_BASE    = 'https://api.maxgood.work'; // single endpoint host per requirements

// -- Activation: create options (autoload=no)
register_activation_hook(__FILE__, function () {
    if (false === get_option(MGW_PUSH_OPTION, false)) {
        add_option(MGW_PUSH_OPTION, [
            'mgoid'    => '',
            'title_id' => '',
            'push_pages'  => 0, // 0/1
       ], '', 'no');
    }
});

// -- Helpers
function mgw_push_get_opts() {
    $opts = get_option(MGW_PUSH_OPTION, []);
    return [
        'mgoid'       => isset($opts['mgoid']) ? (string)$opts['mgoid'] : '',
        'title_id'    => isset($opts['title_id']) ? (string)$opts['title_id'] : '',
        'push_pages'  => isset($opts['push_pages']) ? (int)$opts['push_pages'] : 0,
    ];
}


// -- Admin settings page (simple form, secrets preserved if left blank)
if (is_admin()) {
    add_action('admin_menu', function () {
        add_options_page(
            __('Push to MaxGood.work', 'push-post-to-maxgoodwork'),
            __('Push to MaxGood.work', 'push-post-to-maxgoodwork'),
            'manage_options',
            'mgw-push-settings',
            'mgw_push_render_settings_page'
        );
    });

    add_action('admin_notices', function () {
        if (!current_user_can('manage_options')) return;

        $screen = function_exists('get_current_screen') ? get_current_screen() : null;
        // Show on Dashboard, Posts, Pages, and the plugin's settings page
        $allowed_screens = ['dashboard', 'edit-post', 'edit-page', 'post', 'page', 'settings_page_mgw-push-settings'];
        if ($screen && !in_array($screen->id, $allowed_screens, true)) return;

        $opts = mgw_push_get_opts();
        if (empty($opts['mgoid']) || empty($opts['title_id'])) {
            $settings_url = admin_url('options-general.php?page=mgw-push-settings');
            echo '<div class="notice notice-warning is-dismissible"><p>'
                . esc_html__('Push Post to MaxGood.work:', 'push-post-to-maxgoodwork')
                . ' ' . esc_html__('Missing required settings.', 'push-post-to-maxgoodwork')
                . ' <a href="' . esc_url($settings_url) . '">'
                . esc_html__('Configure now', 'push-post-to-maxgoodwork') . '</a></p></div>';
        }
    });

    function mgw_push_row_action_link($actions, $post) {
        if (!current_user_can('edit_post', $post->ID)) return $actions;
        if (!in_array($post->post_type, ['post','page'], true)) return $actions;

        $url = wp_nonce_url(
            admin_url('admin-post.php?action=mgw_push_now&post=' . (int)$post->ID),
            'mgw_push_now_' . $post->ID
        );

        $actions['mgw_push_now'] = '<a href="' . esc_url($url) . '" aria-label="' . esc_attr__('Push to MaxGood.work now', 'push-post-to-maxgoodwork') . '">'
            . esc_html__('Push to MaxGood.work now', 'push-post-to-maxgoodwork') . '</a>';

        return $actions;
    }
    add_filter('post_row_actions', 'mgw_push_row_action_link', 10, 2);
    add_filter('page_row_actions', 'mgw_push_row_action_link', 10, 2);

    add_action('admin_notices', function () {
        if (!current_user_can('edit_posts')) return;
        if (!isset($_GET['mgw_pushed_now'], $_GET['post']) || (int)$_GET['mgw_pushed_now'] !== 1) return;

        $post_id = (int) $_GET['post'];
        $post = get_post($post_id);
        if (!$post) return;

        echo '<div class="notice notice-success is-dismissible"><p>'
            . sprintf(
                '%s %s',
                esc_html__('Pushed to MaxGood.work:', 'push-post-to-maxgoodwork'),
                '<strong>' . esc_html(get_the_title($post)) . '</strong>'
            )
            . '</p></div>';
    });


    add_action('admin_init', function () {
        register_setting('mgw_push_group', MGW_PUSH_OPTION, [
            'type'              => 'array',
            'sanitize_callback' => function ($input) {
                $out   = [];
                $curr  = get_option(MGW_PUSH_OPTION, []);
                // mgoid: required for sending; treat as secret, preserve if blank
                $mgoid = isset($input['mgoid']) ? trim(wp_unslash($input['mgoid'])) : '';
                $out['mgoid'] = ($mgoid === '') ? (isset($curr['mgoid']) ? $curr['mgoid'] : '') : sanitize_text_field($mgoid);

                // title_id: required; simple slug/id
                $title_id = isset($input['title_id']) ? trim(wp_unslash($input['title_id'])) : '';
                // keep conservative chars
                $title_id = preg_replace('/[^a-zA-Z0-9_\-]/', '', $title_id);
                $out['title_id'] = $title_id;
                // push_pages: checkbox => 1 if present, else 0
                $push_pages = isset($input['push_pages']) ? (int)(bool)$input['push_pages'] : 0;
                $out['push_pages'] = $push_pages;

                return $out;
            },
            'default'           => ['mgoid' => '', 'title_id' => '', 'push_pages' => 0],
            'show_in_rest'      => false,
        ]);
    });

    function mgw_push_render_settings_page() {
        if (!current_user_can('manage_options')) return;
        $opts = mgw_push_get_opts();
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Push Post to MaxGood.work', 'push-post-to-maxgoodwork'); ?></h1>
            <p><?php esc_html_e('When a post is first published, this plugin sends its HTML and metadata to your MaxGood.work Avatar.', 'push-post-to-maxgoodwork'); ?></p>
            <p><em><?php esc_html_e('Note: This plugin only works with existing Avatars/Organizations on MaxGood.work. Visit maxgood.work to learn more and get an Avatar.', 'push-post-to-maxgoodwork'); ?></em></p>
            <form method="post" action="options.php">
                <?php settings_fields('mgw_push_group'); ?>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="mgw_mgoid"><?php esc_html_e('Avatar/Org API Key (mgoid)', 'push-post-to-maxgoodwork'); ?></label></th>
                        <td>
                            <input type="password" id="mgw_mgoid" name="<?php echo esc_attr(MGW_PUSH_OPTION); ?>[mgoid]" value="" size="50" placeholder="<?php echo $opts['mgoid'] ? esc_attr__('**** stored (leave blank to keep)', 'push-post-to-maxgoodwork') : ''; ?>" autocomplete="new-password">
                            <p class="description"><?php esc_html_e('Provided on your MaxGood.work dashboard. Stored securely in the database (not exposed publicly).', 'push-post-to-maxgoodwork'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="mgw_title_id"><?php esc_html_e('Title ID (title_id)', 'push-post-to-maxgoodwork'); ?></label></th>
                        <td>
                            <input type="text" id="mgw_title_id" name="<?php echo esc_attr(MGW_PUSH_OPTION); ?>[title_id]" value="<?php echo esc_attr($opts['title_id']); ?>" size="30" placeholder="e.g. blog2025">
                            <p class="description"><?php esc_html_e('Identifier for the article bundle/content title on MaxGood.work (letters, numbers, underscore, hyphen).', 'push-post-to-maxgoodwork'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="mgw_push_pages"><?php esc_html_e('Push Pages Automatically', 'push-post-to-maxgoodwork'); ?></label></th>
                        <td>
                            <label>
                                <input type="checkbox" id="mgw_push_pages"
                                    name="<?php echo esc_attr(MGW_PUSH_OPTION); ?>[push_pages]"
                                    value="1" <?php checked(1, (int)$opts['push_pages']); ?>>
                                <?php esc_html_e('Automatically push newly published pages (in addition to posts).', 'push-post-to-maxgoodwork'); ?>
                            </label>
                            <p class="description"><?php esc_html_e('If unchecked, only new blog posts are auto-pushed. You can still use the “Push to MaxGood.work now” action on any page.', 'push-post-to-maxgoodwork'); ?></p>
                        </td>
                    </tr>

                </table>
                <?php submit_button(__('Save Settings', 'push-post-to-maxgoodwork')); ?>
            </form>
        </div>
        <?php
    }
}

// -- Publish hook: fire once on first publish
add_action('transition_post_status', function($new, $old, $post) {
    if ($new !== 'publish' || $old === 'publish') return;                // only on first publish
    if (!($post instanceof WP_Post)) return;

    // Only for standard blog posts; optionally pages if enabled
    if ($post->post_type === 'post') {
        // ok
    } elseif ($post->post_type === 'page') {
        $opts = mgw_push_get_opts();
        if (empty($opts['push_pages'])) return; // do not auto-push pages unless enabled
    } else {
        return; // other post types are out of scope
    }

    if (wp_is_post_revision($post->ID) || wp_is_post_autosave($post->ID)) return;
    if (get_post_meta($post->ID, '_mgw_pushed_initial_publish', true)) return;

    mgw_push_send($post);
}, 10, 3);

// -- Manual publish
add_action('admin_post_mgw_push_now', function () {
    if (!isset($_GET['post'])) wp_die('Missing post parameter.');
    $post_id = (int) $_GET['post'];

    if (!current_user_can('edit_post', $post_id)) wp_die('Insufficient permissions.');
    check_admin_referer('mgw_push_now_' . $post_id);

    $post = get_post($post_id);
    if (!$post) wp_die('Invalid post.');

    // Only allow for posts/pages as per requirement
    if (!in_array($post->post_type, ['post', 'page'], true)) wp_die('Unsupported post type.');

    // Attempt send (manual: do NOT set idempotency meta so it can be re-pushed)
    mgw_push_send($post, false);

    // Redirect back to list with a flag for an admin notice
    $referer = wp_get_referer();
    $redirect = $referer ? add_query_arg(['mgw_pushed_now' => 1, 'post' => $post_id], $referer) : admin_url('edit.php');
    wp_safe_redirect($redirect);
    exit;
});


// -- Build minimal HTML content (no theme wrappers; strip binaries)
function mgw_push_render_post_html($post_id) {
    $post = get_post($post_id);
    if (!$post) return '';

    // Ensure global $post is correct so shortcodes/embeds/excerpts render properly
    $prev_global = $GLOBALS['post'] ?? null;
    $GLOBALS['post'] = $post;
    setup_postdata($post);

    // Render content through WP filters
    $html = apply_filters('the_content', $post->post_content);

    // Restore global
    wp_reset_postdata();
    $GLOBALS['post'] = $prev_global;

    // Remove images, videos, audio, iframes, objects, embeds, and data URIs to avoid binaries
    $html = preg_replace('#<(img|video|audio|iframe|object|embed)\b[^>]*>(?:</\1>)?#i', '', $html);
    $html = preg_replace('#(src|href)\s*=\s*([\'"])data:[^\'"]+\2#i', '$1=$2$2', $html);

    // Optionally keep captions as text
    $html = preg_replace('#\[caption[^\]]*\](.*?)\[/caption\]#is', '$1', $html);

    return trim($html);
}


// -- Sender
function mgw_push_send(WP_Post $post, bool $update_meta = true) {
    $opts = mgw_push_get_opts();
    $mgoid = $opts['mgoid'];
    $title_id = $opts['title_id'];

    if ($mgoid === '' || $title_id === '') {
        error_log('[Push to MaxGood] Missing mgoid or title_id; skipping send.');
        return;
    }

    $url = MGW_API_BASE . '/api/upload_content/' . rawurlencode($title_id);

    // Collect metadata
    $author_name = get_the_author_meta('display_name', $post->post_author);
    $permalink   = get_permalink($post);

    // Ensure global $post for excerpt generation as well
    $prev_global = $GLOBALS['post'] ?? null;
    $GLOBALS['post'] = $post;
    setup_postdata($post);

    $excerpt = wp_strip_all_tags(get_the_excerpt($post));

    wp_reset_postdata();
    $GLOBALS['post'] = $prev_global;


    $cats = wp_get_post_terms($post->ID, 'category', ['fields' => 'names']);
    $tags = wp_get_post_terms($post->ID, 'post_tag', ['fields' => 'names']);
    $content_html= mgw_push_render_post_html($post->ID);
    $content_html = '# ' . trim(get_the_title($post)) . "\n\n" . ltrim($content_html);

    $payload = [
        'text' => $content_html,
        'metadata' => [
            'source'       => 'wordpress',
            'event'        => 'post.published',
            'id'          => (string)$post->ID,
            'type'        => $post->post_type,
            'status'      => 'publish',
            'title'       => get_the_title($post),
            'permalink'   => $permalink,
            'author'      => $author_name,
            'publishedAt' => get_post_time('c', true, $post), // ISO8601 UTC
            'categories'  => array_values($cats),
            'tags'        => array_values($tags),
            'excerpt'     => $excerpt,
            'contentHtml' => $content_html, // HTML, no binaries
        ],
    ];

    $body = wp_json_encode($payload, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);

    $headers = [
        'Content-Type' => 'application/json',
        'mgoid'        => $mgoid, // required by API
        'User-Agent'   => 'MG_WP',
    ];

    $args = [
        'headers'     => $headers,
        'body'        => $body,
        'timeout'     => 8,
        'redirection' => 0,
        'blocking'    => false,   // fire-and-forget per “super simple”
        'sslverify'   => true,    // enforce TLS verification
    ];

    $resp = wp_safe_remote_post($url, $args);

    if (is_wp_error($resp)) {
        error_log('[Push to MaxGood] Error: ' . $resp->get_error_message());
        return;
    }

    // mark sent (idempotency) only for automatic first-publish flow
    if ($update_meta) {
        update_post_meta($post->ID, '_mgw_pushed_initial_publish', current_time('mysql', true));
    }
}
